//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.io;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.component.LifeCycle;
import org.eclipse.jetty.util.thread.ScheduledExecutorScheduler;
import org.eclipse.jetty.util.thread.Scheduler;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertSame;
import static org.junit.jupiter.api.Assertions.assertTrue;

public class CyclicTimeoutsTest
{
    private Scheduler scheduler;
    private CyclicTimeouts<CyclicTimeouts.Expirable> timeouts;

    @BeforeEach
    public void prepare()
    {
        scheduler = new ScheduledExecutorScheduler();
        LifeCycle.start(scheduler);
    }

    @AfterEach
    public void dispose()
    {
        if (timeouts != null)
            timeouts.destroy();
        LifeCycle.stop(scheduler);
    }

    @Test
    public void testNoExpirationForNonExpiringEntity() throws Exception
    {
        CountDownLatch latch = new CountDownLatch(1);
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                latch.countDown();
                return null;
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                return false;
            }
        };

        // Schedule an entity that does not expire.
        timeouts.schedule(ConstantExpirable.noExpire());

        Assertions.assertFalse(latch.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void testExpirableEntityBecomesNonExpirable() throws Exception
    {
        long timeout = 1000;
        DynamicExpirable entity = new DynamicExpirable(NanoTime.now() + TimeUnit.MILLISECONDS.toNanos(timeout));
        CountDownLatch latch = new CountDownLatch(1);
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                entity.expireNanoTime = Long.MAX_VALUE;
                return List.<Expirable>of(entity).iterator();
            }

            @Override
            boolean schedule(CyclicTimeout cyclicTimeout, long delay, TimeUnit unit)
            {
                if (unit.toMillis(delay) > 2 * timeout)
                    latch.countDown();
                return super.schedule(cyclicTimeout, delay, unit);
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                latch.countDown();
                return false;
            }
        };

        timeouts.schedule(entity);

        // Wait until the timeouts check.
        Thread.sleep(timeout);

        // Since the expireNanoTime was changed to Long.MAX_VALUE,
        // the entity must not have been scheduled nor expired.
        Assertions.assertFalse(latch.await(1, TimeUnit.SECONDS));
    }

    @Test
    public void testScheduleZero() throws Exception
    {
        ConstantExpirable entity = ConstantExpirable.ofDelay(0, TimeUnit.SECONDS);
        CountDownLatch iteratorLatch = new CountDownLatch(1);
        CountDownLatch expiredLatch = new CountDownLatch(1);
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                iteratorLatch.countDown();
                return Collections.emptyIterator();
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                expiredLatch.countDown();
                return false;
            }
        };

        timeouts.schedule(entity);

        Assertions.assertTrue(iteratorLatch.await(1, TimeUnit.SECONDS));
        Assertions.assertFalse(expiredLatch.await(1, TimeUnit.SECONDS));
    }

    @ParameterizedTest
    @ValueSource(booleans = {false, true})
    public void testIterateAndExpire(boolean remove) throws Exception
    {
        ConstantExpirable zero = ConstantExpirable.ofDelay(0, TimeUnit.SECONDS);
        ConstantExpirable one = ConstantExpirable.ofDelay(1, TimeUnit.SECONDS);
        Collection<CyclicTimeouts.Expirable> collection = new ArrayList<>();
        collection.add(one);
        AtomicInteger iterations = new AtomicInteger();
        CountDownLatch expiredLatch = new CountDownLatch(1);
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                iterations.incrementAndGet();
                return collection.iterator();
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                assertSame(one, expirable);
                expiredLatch.countDown();
                return remove;
            }
        };

        // Triggers immediate call to iterator(), which
        // returns an entity that expires in 1 second.
        timeouts.schedule(zero);

        // After 1 second there is a second call to
        // iterator(), which returns the now expired
        // entity, which is passed to onExpired().
        assertTrue(expiredLatch.await(2, TimeUnit.SECONDS));

        // Wait for the collection to be processed
        // with the return value of onExpired().
        Thread.sleep(1000);

        // Verify the processing of the return value of onExpired().
        assertEquals(remove ? 0 : 1, collection.size());

        // Wait to see if iterator() is called again (it should not).
        Thread.sleep(1000);
        assertEquals(2, iterations.get());
    }

    @Test
    public void testScheduleOvertake() throws Exception
    {
        ConstantExpirable zero = ConstantExpirable.ofDelay(0, TimeUnit.SECONDS);
        long delayMs = 2000;
        ConstantExpirable two = ConstantExpirable.ofDelay(delayMs, TimeUnit.MILLISECONDS);
        ConstantExpirable overtake = ConstantExpirable.ofDelay(delayMs / 2, TimeUnit.MILLISECONDS);
        Collection<CyclicTimeouts.Expirable> collection = new ArrayList<>();
        collection.add(two);
        CountDownLatch expiredLatch = new CountDownLatch(2);
        List<CyclicTimeouts.Expirable> expired = new ArrayList<>();
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            private final AtomicBoolean overtakeScheduled = new AtomicBoolean();

            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                return collection.iterator();
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                expired.add(expirable);
                expiredLatch.countDown();
                return true;
            }

            @Override
            boolean schedule(CyclicTimeout cyclicTimeout, long delay, TimeUnit unit)
            {
                if (delay <= 0)
                    return super.schedule(cyclicTimeout, delay, unit);

                // Simulate that an entity with a shorter timeout
                // overtakes the entity that is currently being scheduled.
                // Only schedule the overtake once.
                if (overtakeScheduled.compareAndSet(false, true))
                {
                    collection.add(overtake);
                    schedule(overtake);
                }
                return super.schedule(cyclicTimeout, delay, unit);
            }
        };

        // Trigger the initial call to iterator().
        timeouts.schedule(zero);

        // Make sure that the overtake entity expires properly.
        assertTrue(expiredLatch.await(2 * delayMs, TimeUnit.MILLISECONDS));

        // Make sure all entities expired properly.
        assertSame(overtake, expired.get(0));
        assertSame(two, expired.get(1));
    }

    @Test
    public void testDynamicExpirableEntityIsNotifiedMultipleTimes() throws Exception
    {
        long delay = 500;
        DynamicExpirable entity = new DynamicExpirable(NanoTime.now() + TimeUnit.MILLISECONDS.toNanos(delay));
        List<CyclicTimeouts.Expirable> entities = List.of(entity);

        CountDownLatch latch = new CountDownLatch(2);
        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<CyclicTimeouts.Expirable> iterator()
            {
                return entities.iterator();
            }

            @Override
            protected boolean onExpired(CyclicTimeouts.Expirable expirable)
            {
                assertSame(entity, expirable);
                // Postpone expiration.
                entity.expireNanoTime = NanoTime.now() + TimeUnit.MILLISECONDS.toNanos(delay);
                latch.countDown();
                return false;
            }
        };

        // Trigger the initial call to iterator().
        timeouts.schedule(entities.get(0));

        assertTrue(latch.await(3 * delay, TimeUnit.MILLISECONDS), latch.toString());
    }

    @Test
    public void testExpiredEntityIsNotRescheduled() throws Exception
    {
        long timeout = 500;
        DynamicExpirable entity = new DynamicExpirable(NanoTime.now() + TimeUnit.MILLISECONDS.toNanos(timeout));
        List<CyclicTimeouts.Expirable> entities = List.of(entity);

        List<Long> schedules = new ArrayList<>();

        timeouts = new CyclicTimeouts<>(scheduler)
        {
            @Override
            protected Iterator<Expirable> iterator()
            {
                return entities.iterator();
            }

            @Override
            protected boolean onExpired(Expirable expirable)
            {
                // Reset the expiration, but do not remove the entity.
                entity.expireNanoTime = Long.MAX_VALUE;
                return false;
            }

            @Override
            boolean schedule(CyclicTimeout cyclicTimeout, long delay, TimeUnit unit)
            {
                schedules.add(unit.toMillis(delay));
                return super.schedule(cyclicTimeout, delay, unit);
            }
        };

        // Trigger the initial call to iterator().
        timeouts.schedule(entities.get(0));

        // Let the entity expire.
        Thread.sleep(2 * timeout);

        // Verify that after expiration the entity has not been rescheduled.
        assertThat(schedules.toString(), schedules.size(), is(1));
    }

    private static class ConstantExpirable implements CyclicTimeouts.Expirable
    {
        private static ConstantExpirable noExpire()
        {
            return new ConstantExpirable();
        }

        private static ConstantExpirable ofDelay(long delay, TimeUnit unit)
        {
            return new ConstantExpirable(delay, unit);
        }

        private final long expireNanoTime;
        private final String asString;

        private ConstantExpirable()
        {
            this.expireNanoTime = Long.MAX_VALUE;
            this.asString = "noexp";
        }

        public ConstantExpirable(long delay, TimeUnit unit)
        {
            this.expireNanoTime = NanoTime.now() + unit.toNanos(delay);
            this.asString = String.valueOf(unit.toMillis(delay));
        }

        @Override
        public long getExpireNanoTime()
        {
            return expireNanoTime;
        }

        @Override
        public String toString()
        {
            return String.format("%s@%x[%sms]", getClass().getSimpleName(), hashCode(), asString);
        }
    }

    private static class DynamicExpirable implements CyclicTimeouts.Expirable
    {
        private long expireNanoTime;

        public DynamicExpirable(long expireNanoTime)
        {
            this.expireNanoTime = expireNanoTime;
        }

        @Override
        public long getExpireNanoTime()
        {
            return expireNanoTime;
        }

        @Override
        public String toString()
        {
            return String.format("%s@%x[%dms]", getClass().getSimpleName(), hashCode(), NanoTime.millisUntil(expireNanoTime));
        }
    }
}
